/*
 * This File is one of the examples from Java For Beginners
 * Do not reproduce this code for others or use it in a commercial setting without prior permission from the author.
 */
package chapter9.datetime;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.Month;
import static java.time.temporal.ChronoUnit.YEARS;

/**
 * @author Matthew Gregory website: www.matthewgregory-author.com.au
 *
 */
public class Person {

    public String firstName;
    public String lastName;
    public String type;
    public LocalDate birthday;
    public LocalDateTime reviewDate;

    //Constructor
    public Person(String type, String firstName, String lastName, LocalDate birthday) {
        this.type = checkType(type);
        setNames(firstName, lastName);
        this.birthday = birthday;
        setReviewDate();
    }

    public Person() {
    }

    public String getFullName() {
        return Utilities.capitalise(firstName + " " + lastName);
    }

    public long getAge() {
        //What is the date today?
        LocalDate today = LocalDate.now();
        //What is the difference between today's date and birthday
        long age = YEARS.between(birthday, today);
        return age;
    }

    //This method sets the review date to the first day the next financial year.
    //If it not a weekday then it is adjusted to the next Monday.
    public void setReviewDate() {
        //Ignore if the Person is not an employee
        if (!this.type.equals("Employee")) {
            return;
        }
        //Set the review date to the 30th June this year.
        LocalDateTime newReviewDate = LocalDateTime.of(
                //Current Year
                LocalDate.now().getYear(),
                //June 30th
                Month.JUNE, 30,
                //Leave the time at midnight for now
                0, 0);
        //If this date has already occurted then we need to add a year
        if (newReviewDate.isBefore(LocalDateTime.now())){
            newReviewDate.plusYears(1);
        }
        
        //If the date is a Saturday or Sunday the adjust it accordingly
        switch (newReviewDate.getDayOfWeek().name()){
            case "Saturday"://If the day for the week is a Saturday, add 2 days
                newReviewDate.plusDays(2);
                break;
            case "Sunday"://If the day for the week is a Sunday, add 1 days
                newReviewDate.plusDays(1);
                break;
        }

        this.reviewDate = newReviewDate;
    }
    
    //This method can be used to set the meeting time for our review date.
    public void setReviewTime(int hours,int mins){
        //Change the time of our review date.
        this.reviewDate=this.reviewDate.withHour(hours).withMinute(mins);
        System.out.println("Time has been successfully changed to: "+reviewDate.toLocalTime());
    }

    public void printDetails() {
        System.out.println("First Name: " + firstName);
        System.out.println("Last Name: " + lastName);
        System.out.println("Full Name: " + getFullName());
//        System.out.println("Birthday: "+birthday);
        System.out.println("Age: " + getAge());
        System.out.println("Review Date: " + reviewDate);
    }

    public void setNames(String firstName, String lastName) {
        this.firstName = firstName.toUpperCase();
        this.lastName = lastName.toUpperCase();
    }

    public String checkType(String type) {
        //Using exceptions to make sure type is either employee or customer.
        if (type != "Employee" && type != "Customer") {
            throw new RuntimeException("Warning you have specified a type of " + type + ". Type should be either Employee or Customer");
        } else {
            return type;
        }
    }
    
    public String toString(){
        return getFullName();
    }

}
